import {CosmosBlock, CosmosMessage, CosmosTransaction,} from "@subql/types-cosmos";
import {Block, BlockSignature, Event, Message, Transaction} from "../types";
import {toBech32} from "@cosmjs/encoding";
import {createHash} from 'crypto';
import {BinaryLike} from "node:crypto";
import {validatorMap} from "../validators";

declare global {
    interface BigInt {
        toJSON(): Number;
    }
}
BigInt.prototype.toJSON = function () {
    return Number(this);
};

export async function handleBlock(block: CosmosBlock): Promise<void> {
    try {
        const blockHash = block.blockId?.hash
            ? bytesToHex(block.blockId.hash)
            : `missing_hash_${block.header.height}`;

        const blockHeight = BigInt(block.header.height);

        const blockRecord = Block.create({
            id: blockHeight.toString(),
            time: new Date(block.header.time.toISOString()),
            hash: blockHash,
            blockHeight,
            proposer: getValidatorAddress(block.header.proposerAddress),
            chainId: block.header.chainId,
            rawBlock: JSON.stringify(block),
        });

        await blockRecord.save();

        const signatures = block.block.lastCommit?.signatures ?? [];
        await Promise.all(
            signatures.map((sig, index) => {
                const validatorConsensusAddress = getValidatorAddress(sig?.validatorAddress)
                if (!validatorConsensusAddress) {
                    console.warn(`Invalid validator address for signature at index ${index} in block #${block.header.height}, JSON: ${JSON.stringify(sig)}`);
                    return Promise.resolve();
                }
                const signatureRecord = BlockSignature.create({
                    id: `${blockHeight.toString()}-${validatorConsensusAddress}`,
                    blockHeight,
                    validatorConsensusAddress,
                    signed: !!sig.signature,
                });

                return signatureRecord.save();
            })
        );

        logger.info(`📦 Processed block #${block.header.height} sig:${signatures.length}`);
    } catch (e: any) {
        logger.error(`Error processing block #${block.header.height}: ${e.message}`);
    }
}

export async function handleTransaction(tx: CosmosTransaction, test: any, test2: any): Promise<void> {
    // logger.info(`💸 New Transaction: ${tx.hash} in block ${tx.block.header.height}`);

    const txHash = hashIt(tx.block.block.txs[tx.idx]);
    const blockId = tx.block.header.height.toString();
    const blockHeight = BigInt(tx.block.header.height);
    const timestamp = new Date(tx.block.header.time as unknown as string);

    const decoded = tx.decodedTx;

    const fee = decoded?.authInfo?.fee?.amount?.[0];
    const gasWanted = BigInt(tx.tx?.gasWanted ?? 0);
    const gasUsed = BigInt(tx.tx?.gasUsed ?? 0);
    const code = tx.tx?.code ?? 0;

    const txType = decoded?.body?.messages?.map((msg: any) => transformMessageName(msg.typeUrl)).join("|") || "unknown";

    const transaction = Transaction.create({
        id: txHash,
        blockId,
        blockHeight,
        txHash,
        txType,
        memo: tx.decodedTx?.body?.memo || "",
        rawTransaction: JSON.stringify(deepReplaceBuffers(tx.tx)),
        feeAmount: BigInt(fee?.amount || 0),
        feeDenom: fee?.denom || "",
        gasWanted,
        gasUsed,
        code,
        rawLog: tx.tx?.log ?? "",
        timestamp,
    });
    await transaction.save();

    // 📡 Events
    const events = tx.tx?.events || [];
    for (const [i, event] of events.entries()) {
        for (const attr of event.attributes || []) {
            const id = `${txHash}-${i}-${attr.key}`;
            const ev = Event.create({
                id,
                transactionId: txHash,
                eventType: event.type,
                attrKey: attr.key.toString(),
                attrValue: attr.value.toString(),
                attrIndexed: false,
                createdAt: timestamp,
            });
            await ev.save();
        }
    }
    logger.info(`💸 Saved transaction ${txHash}`);
}

export async function handleMessage(message: CosmosMessage): Promise<void> {
    // logger.info(`Msg tx hash: ${message.tx.hash} ${JSON.stringify(message)}`);
    const txHash = hashIt(message.block.block.txs[message.tx.idx]);
    const mess = Message.create({
        id: `${txHash}-${message.idx}`,
        transactionId: txHash,
        msgIndex: message.idx,
        typeUrl: message.msg.typeUrl,
        jsonBody: JSON.stringify(message.msg),
    });
    await mess.save();
}

const hashIt = (bytes: BinaryLike): string =>
    createHash('sha256').update(bytes).digest('hex').toUpperCase();

function bytesToHex(bytes: Uint8Array): string {
    return Buffer.from(bytes).toString("hex").toUpperCase();
}

function transformMessageName(input: string): string {
    if (!input) return "Unknown";
    const lastSegment = input.split('.').pop() || '';
    const nameWithoutMsg = lastSegment.startsWith('Msg') ? lastSegment.slice(3) : lastSegment;
    return nameWithoutMsg.replace(/([A-Z])/g, ' $1').trim();
}

function deepReplaceBuffers(value: any): any {
    if (Buffer.isBuffer(value)) {
        return value.toString("base64");
    }

    if (Array.isArray(value)) {
        return value.map(deepReplaceBuffers);
    }

    if (value && typeof value === "object" && !(value instanceof Date)) {
        const replaced: Record<string, any> = {};
        for (const [k, v] of Object.entries(value)) {
            replaced[k] = deepReplaceBuffers(v);
        }
        return replaced;
    }

    return value;
}

function getValidatorAddress(raw: any): string {
    if (!raw) return 'unknown';

    const bytes = typeof raw === 'string' ? Buffer.from(raw, 'hex') : raw;
    if (bytes.length !== 20) return `invalid_length_${bytes.length}`;

    const consensusAddress = toBech32('cosmosvalcons', bytes);
    return validatorMap[consensusAddress];
}