import * as fs from 'fs';
import * as path from 'path';

const rootPath = process.cwd();
const ROOT_PROTO_DIR = path.join(rootPath, 'proto');
const OUT_FILE = path.join(rootPath, 'src', 'chaintypes.ts');

const IGNORED = [
    '/google/protobuf/descriptor.proto',
    // при желании можно добавить и другие «невостребованные» proto
];


/**
 * Рекурсивно собирает все .proto-файлы из заданной директории
 */
function findAllProtoFiles(dir: string): string[] {
    let results: string[] = [];
    for (const name of fs.readdirSync(dir)) {
        const full = path.join(dir, name);
        // console.log("[SSA] ", `src/tools/gen-chaintypes.ts:21 name:`, full);
        if (fs.statSync(full).isDirectory()) {
            results = results.concat(findAllProtoFiles(full));
        } else if (
            name.endsWith('.proto') &&
            !IGNORED.some(ignored => full.endsWith(ignored))
        ) {
            console.log("[SSA] ", `src/tools/gen-chaintypes.ts:28 full:`, full);
            results.push(full);
        }
    }
    return results;
}

/**
 * Извлекает все определения message из одного .proto
 */
function extractMessages(filePath: string): string[] {
    const content = fs.readFileSync(filePath, 'utf8');
    const regex = /^\s*message\s+([A-Za-z0-9_]+)/gm;
    const messages: string[] = [];
    let m: RegExpExecArray | null;
    while (m = regex.exec(content)) {
        messages.push(m[1]);
    }
    return messages;
}

/**
 * Преобразует абсолютный путь вида
 *   /…/project/proto/cosmos/bank/v1beta1/tx.proto
 * в строку-namespace
 *   "cosmos.bank.v1beta1.tx"
 */
function toNamespace(filePath: string): string {
    const rel = path.relative(ROOT_PROTO_DIR, filePath);
    return rel
        .replace(/\.proto$/, '')    // убрать расширение
        .split(path.sep)            // на части по папкам
        .join('.');                 // соединить точками
}

/**
 * Собирает финальный код
 */
function buildChaintypesMap(protoFiles: string[]): string {
    const entries: string[] = [];

    for (const file of protoFiles) {
        const msgs = extractMessages(file);
        if (msgs.length === 0) continue;

        const ns = toNamespace(file);
        const relImport = './proto/' + path.relative(path.resolve(__dirname, '..'), file).replace(/\\/g, '/');
        entries.push(`  [
    "${ns}",
    { file: "${relImport}", messages: [${msgs.sort().map(m => `"${m}"`).join(', ')}] }
  ]`);
    }

    return `// AUTO-GENERATED by generate-chaintypes-map.ts — do not edit
export const chaintypes = new Map<string, { file: string; messages: string[] }>([
${entries.join(',\n')}
]);
`;
}

/**
 * main
 */
function run() {
    const allProto = findAllProtoFiles(ROOT_PROTO_DIR);
    const code = buildChaintypesMap(allProto);
    fs.writeFileSync(OUT_FILE, code, 'utf8');
    console.log(`✅ ${path.basename(OUT_FILE)} generated, entries: ${allProto.length}`);
}

run();