import fs from "fs";
import axios from "axios";
import {fromBase64, toBech32} from "@cosmjs/encoding";
import dotenv from "dotenv";
import {sha256} from "@noble/hashes/sha256";
import path from "path";

dotenv.config();

const rootPath = process.cwd();
const outputFile = path.join(rootPath, 'src', 'validators.ts');

async function genValidators(): Promise<Record<string, string>> {
    let nextKey: string | undefined = undefined;
    let allValidators: any[] = [];

    do {
        const res: any = await axios.get(
            `${process.env.API_ENDPOINT}/cosmos/staking/v1beta1/validators`,
            {
                params: {
                    "pagination.limit": 1000,
                    "pagination.key": nextKey,
                },
            }
        );

        const { validators, pagination } = res.data;
        allValidators = allValidators.concat(validators);
        nextKey = pagination?.next_key || undefined;
    } while (nextKey);

    const map: Record<string, string> = {};

    for (const val of allValidators) {
        const key = val.consensus_pubkey?.key;
        if (!key) continue;

        try {
            const pubkeyBytes = fromBase64(key);
            const hash = sha256(pubkeyBytes).slice(0, 20); // именно это — validatorAddress
            const consensusAddr = toBech32('cosmosvalcons', hash);
            map[consensusAddr] = val.operator_address;
        } catch (e) {
            console.warn(`Invalid consensus pubkey for validator ${val.operator_address}`);
        }
    }

    return map;
}

async function generateTS() {
    const map = await genValidators();

    const content = `// AUTO-GENERATED FILE — do not edit manually
export const validatorMap: Record<string, string> = ${JSON.stringify(map, null, 2)};
`;

    fs.writeFileSync(outputFile, content);
    console.log(`✅ Generated validator-map.ts with ${Object.keys(map).length} entries`);
}

generateTS();